/**
 * Generic Transport for the browser, using the XmlHttpRequest object
 *
 * @class  connections.Xhr
 */
module.exports = XhrConnector;

/* jshint browser:true */

var _ = require('../utils');
var ConnectionAbstract = require('../connection');
var ConnectionFault = require('../errors').ConnectionFault;
var asyncDefault = !(navigator && /PhantomJS/i.test(navigator.userAgent));

function XhrConnector(host, config) {
  ConnectionAbstract.call(this, host, config);
}
_.inherits(XhrConnector, ConnectionAbstract);

/**
 * Simply returns an XHR object cross browser
 * @type {Function}
 */
var getXhr = _.noop;

if (typeof XMLHttpRequest !== 'undefined') {
  // rewrite the getXhr method to always return the native implementation
  getXhr = function () {
    return new XMLHttpRequest();
  };
} else {
  // find the first MS implementation available
  getXhr = _.first(['Msxml2.XMLHTTP', 'Microsoft.XMLHTTP', 'Msxml2.XMLHTTP.4.0'], function (appName) {
    /* jshint unused: false */
    try {
      var test = new window.ActiveXObject(appName);
      return function () {
        return new window.ActiveXObject(appName);
      };
    } catch (e) {
      return null;
    }
  });
}

if (!getXhr) {
  throw new Error('getXhr(): XMLHttpRequest not available');
}

XhrConnector.prototype.request = function (params, cb) {
  var xhr = getXhr();
  var timeoutId;
  var url = this.host.makeUrl(params);
  var headers = this.host.getHeaders(params.headers);

  var log = this.log;
  var async = params.async === false ? false : asyncDefault;

  if (params.auth) {
    xhr.open(params.method || 'GET', url, async, params.auth.user, params.auth.pass);
  } else {
    xhr.open(params.method || 'GET', url, async);
  }

  xhr.onreadystatechange = function () {
    if (xhr.readyState === 4) {
      clearTimeout(timeoutId);
      log.trace(params.method, url, params.body, xhr.responseText, xhr.status);
      var err = xhr.status ? void 0 : new ConnectionFault(xhr.statusText || 'Request failed to complete.');
      cb(err, xhr.responseText, xhr.status);
    }
  };

  if (headers) {
    for (var key in headers) {
      if (headers[key] !== void 0) {
        xhr.setRequestHeader(key, headers[key]);
      }
    }
  }

  xhr.send(params.body || void 0);

  return function () {
    xhr.abort();
  };
};
